import torch.nn as nn
import torch
from spikingjelly.clock_driven import layer
from search.neuron_operations import *
from search.super_model import operations, OP, cfg

__all__ = ['spiking_resnet18', 'spiking_resnet34', 'spiking_vgg11', 'spiking_vgg13', 'spiking_vgg16', 'spiking_vgg19']


class SpikingVGG(nn.Module):
    def __init__(self, vgg_name, neuron, dropout=0.0, num_classes=10, **kwargs):
        super(SpikingVGG, self).__init__()
        self.whether_bias = True
        self.init_channels = kwargs.get('c_in', 2)
        self.layers = self._count_layers(cfg[vgg_name])
        self.neurons = self.cut_neruons(neuron, self.layers)

        # here neuron should be cut as vgg_name's length
        self.layer1 = self._make_layers(cfg[vgg_name][0], dropout, self.neurons[0], **kwargs)
        self.layer2 = self._make_layers(cfg[vgg_name][1], dropout, self.neurons[1], **kwargs)
        self.layer3 = self._make_layers(cfg[vgg_name][2], dropout, self.neurons[2], **kwargs)
        self.layer4 = self._make_layers(cfg[vgg_name][3], dropout, self.neurons[3], **kwargs)
        self.layer5 = self._make_layers(cfg[vgg_name][4], dropout, self.neurons[4], **kwargs)
        self.avgpool = nn.AdaptiveAvgPool2d((7, 7))
        self.classifier = nn.Sequential(
            nn.Flatten(),
            nn.Linear(512 * 7 * 7, num_classes),
        )

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                if m.bias is not None:
                    nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.Linear):
                nn.init.normal_(m.weight, 0, 0.01)
                nn.init.constant_(m.bias, 0)

    def _make_layers(self, cfg, dropout, neuron, **kwargs):
        layers = []
        count = 0
        for x in cfg:
            if x == 'M':
                layers.append(nn.AvgPool2d(kernel_size=2, stride=2))
            else:
                layers.append(nn.Conv2d(self.init_channels, x, kernel_size=3, padding=1, bias=self.whether_bias))
                layers.append(nn.BatchNorm2d(x))
                layers.append(operations[OP[neuron[count]]](**kwargs))
                layers.append(layer.Dropout(dropout))
                count += 1
                self.init_channels = x
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.layer1(x)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = self.layer5(out)
        out = self.avgpool(out)
        out = self.classifier(out)
        return out


    def _count_layers(self, cfg):
        # count here is different from the count in super_model.py
        cumulative_counts = []
        for layer in cfg:
            count = 0
            count += sum(1 for x in layer if not isinstance(x, str))
            cumulative_counts.append(count)
        return cumulative_counts

    def cut_neruons(self, neurons, counts):
        # cut neurons
        splits = []
        start = 0
        for length in counts:
            end = start + length
            splits.append(neurons[start:end])
            start = end
        return splits

class ResnetBlock(nn.Module):
    expansion = 1

    def __init__(self, in_channels, out_channels, stride, neuron, dropout, **kwargs):
        super(ResnetBlock, self).__init__()
        whether_bias = True

        # input neuron: a list with shape (1,2), dim=0 is the sn1's neuron, dim=1 is the sn2's neuron
        self.bn1 = nn.BatchNorm2d(in_channels)
        self.conv1 = nn.Conv2d(in_channels, out_channels, kernel_size=3, stride=stride, padding=1, bias=whether_bias)
        self.bn2 = nn.BatchNorm2d(out_channels)
        self.dropout = layer.Dropout(dropout)
        self.conv2 = nn.Conv2d(out_channels, self.expansion * out_channels, kernel_size=3, stride=1, padding=1, bias=whether_bias)

        if stride != 1 or in_channels != self.expansion * out_channels:
            self.shortcut = nn.Conv2d(in_channels, self.expansion * out_channels, kernel_size=1, stride=stride, padding=0, bias=whether_bias)
        else:
            self.shortcut = nn.Sequential()

        self.sn1 = operations[OP[neuron[0]]](**kwargs)
        self.sn2 = operations[OP[neuron[1]]](**kwargs)

    def forward(self, x):
        x = self.sn1(self.bn1(x))
        out = self.conv1(x)
        out = self.conv2(self.dropout(self.sn2(self.bn2(out))))
        out = out + self.shortcut(x)
        return out


class SpikingResnet(nn.Module):

    def __init__(self, block, num_blocks, optimal_neuron, num_classes, dropout, **kwargs):
        super(SpikingResnet, self).__init__()
        self.data_channels = kwargs.get('c_in', 3)
        self.init_channels = 64

        self.conv1 = nn.Conv2d(self.data_channels, 64, kernel_size=3, stride=1, padding=1, bias=False)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], 1, optimal_neuron[0:num_blocks[0] * 2],  dropout, **kwargs)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], 2, optimal_neuron[num_blocks[0] * 2:(num_blocks[0] + num_blocks[1]) * 2], dropout, **kwargs)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], 2, optimal_neuron[(num_blocks[0] + num_blocks[1]) * 2:(num_blocks[0] + num_blocks[1] + num_blocks[2]) * 2], dropout, **kwargs)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], 2, optimal_neuron[(num_blocks[0] + num_blocks[1] + num_blocks[2]) * 2:(num_blocks[0] + num_blocks[1] + num_blocks[2] + num_blocks[3]) * 2], dropout, **kwargs)
        self.bn1 = nn.BatchNorm2d(512 * block.expansion)
        self.pool = nn.AvgPool2d(4)
        self.flat = nn.Flatten()
        self.drop = layer.Dropout(dropout)
        self.linear = nn.Linear(512 * block.expansion, num_classes)

        # LIF neuron
        self.relu1 = operations[OP[optimal_neuron[-1]]](**kwargs)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, val=1)
                nn.init.zeros_(m.bias)
            elif isinstance(m, nn.Linear):
                nn.init.zeros_(m.bias)

    def _make_layer(self, block, out_channels, num_blocks, stride, neuron, dropout, **kwargs):
        # here neuron [x,x,x,x]
        strides = [stride] + [1] * (num_blocks-1)
        layers = []
        index = 0
        for stride in strides:
            # each time give [x,x] of neuron
            layers.append(block(self.init_channels, out_channels, stride, neuron[index*2:index*2+2], dropout,  **kwargs))
            self.init_channels = out_channels * block.expansion
            index = index + 1
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.conv1(x)
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = self.pool(self.relu1(self.bn1(out)))
        out = self.drop(self.flat(out))
        out = self.linear(out)
        return out


def spiking_resnet18(optimal_neuron, num_classes=10, neuron_dropout=0.0, **kwargs):
    return SpikingResnet(ResnetBlock, [2, 2, 2, 2], optimal_neuron, num_classes, neuron_dropout,  **kwargs)


def spiking_resnet34(optimal_neuron, num_classes=10, neuron_dropout=0.0, **kwargs):
    return SpikingResnet(ResnetBlock, [3, 4, 6, 3], optimal_neuron, num_classes, neuron_dropout,  **kwargs)


def spiking_vgg11(optimal_neuron, num_classes=10, neuron_dropout=0.0, **kwargs):
    return SpikingVGG('VGG11', neuron=optimal_neuron, dropout=neuron_dropout, num_classes=num_classes, **kwargs)

def spiking_vgg13(optimal_neuron, num_classes=10, neuron_dropout=0.0, **kwargs):
    return SpikingVGG('VGG13', neuron=optimal_neuron, dropout=neuron_dropout, num_classes=num_classes, **kwargs)

def spiking_vgg16(optimal_neuron, num_classes=10, neuron_dropout=0.0, **kwargs):
    return SpikingVGG('VGG16', neuron=optimal_neuron, dropout=neuron_dropout, num_classes=num_classes, **kwargs)

def spiking_vgg19(optimal_neuron, num_classes=10, neuron_dropout=0.0, **kwargs):
    return SpikingVGG('VGG19', neuron=optimal_neuron, dropout=neuron_dropout, num_classes=num_classes, **kwargs)